#Copyright 2018 Streamline Technologies, Inc.
#All Rights Reserved
#Name:  Frank W. McKinnie, PE, CFM
#Company: Streamline Technologies, Inc.
#Purpose:  Writes CSV for import into ICPR4 from GWIS version 2.1 GDB

def Coves(OFRegionDict,outpath, stem):

    import arcpy
    from iccsv import csv_writerow
    
    ####################################################################
    #Determine Cove information
    ####################################################################
    try:
        
        infc = "OF_Cove"
        
        #Clear current selection
        try:
            arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
        except:            
            pass

        # Identify the input feature class to describe
        #
        desc = arcpy.Describe(infc)
        shapefieldname = desc.ShapeFieldName

        # Create search cursor
        #
        rows = arcpy.SearchCursor(infc)

        #Initial flag values
        Cove_Info = []        

        # Enter for loop for each feature/row
        for row in rows:

            # Create empty list/tuple for points
            Cove_PTs = []
            
            #Flag to include coves that are assigned to regions included in GDB
            flag = False
            if row.OverlandFlowRegion in OFRegionDict:
                flag = True

            #Process coves with regions in GDB
            if flag == True:

                # Create the geometry object
                #
                feat = row.getValue(shapefieldname)
                partnum = 0

                # Count the number of points in the current multipart feature
                #
                partcount = feat.partCount

                #Total number of points in feature    
                pointcount = feat.pointCount        

                # Enter while loop for each part in the feature (if a singlepart feature
                # this will occur only once)
                #
                while partnum < partcount:
                    # Print the part number
                    #
                    part = feat.getPart(partnum)
                    pnt = part.next()
                    pntcount = 0

                    # Enter while loop for each vertex
                    #
                    while pnt:                    

                        #Add cove data to tuple
                        Cove_Data = (pntcount, pnt.X, pnt.Y)
                        Cove_PTs.append(Cove_Data)
                                    
                        pnt = part.next()
                        pntcount = pntcount + 1

                        # If pnt is null, either the part is finished or there is an 
                        #   interior ring
                        #
                        if not pnt: 
                            pnt = part.next()
                            if pnt:
                                print("Interior Ring:")
                    partnum = partnum + 1
                    
                #Determine first and last point coordinates for cove line and direction drawn                
                n = pntcount            
                i = 0
                SUM = 0

                while i < n:
                    
                    if i < n-1:
                        
                        Xi0 = Cove_PTs[i][1]
                        Xi1 = Cove_PTs[i+1][1]
                        Yi0 = Cove_PTs[i][2]
                        Yi1 = Cove_PTs[i+1][2]

                        delta_SUM = (Xi1-Xi0)*(Yi1+Yi0)
                        SUM = SUM + delta_SUM

                        i = i + 1

                    if i == n-1:
                       
                        Xi0 = Cove_PTs[i][1]
                        Xi1 = Cove_PTs[0][1]
                        Yi0 = Cove_PTs[i][2]
                        Yi1 = Cove_PTs[0][2]

                        delta_SUM = (Xi1-Xi0)*(Yi1+Yi0)
                        SUM = SUM + delta_SUM
                        i = i + 1            
                       

                #Determine direction
                if SUM < 0:
                    Direction = "COUNTERCLOCKWISE"
                else:
                    Direction = "CLOCKWISE"
                   
                #Create tuple with cove line data (GUID, CV name, CoveName, StartPointX, EndPointX, StartPointY, EndPointY, Direction)
                Cove_Data = (row.ICPR_GUID,
                             row.ControlVolumeName,
                             row.CoveName,
                             Cove_PTs[0][1],
                             Cove_PTs[n-1][1],
                             Cove_PTs[0][2],
                             Cove_PTs[n-1][2],
                             Direction)
                
                Cove_Info.append(Cove_Data)

        #empty cursor
        del rows

    except:        
        arcpy.GetMessages()

    ####################################################################
    #Determine Cove Point information
    ####################################################################
    try:
        
        infc = "OF_Cove_Point"

        #Clear current selection
        try:
            arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
        except:            
            pass

        # Identify the input feature class to describe
        #
        desc = arcpy.Describe(infc)
        shapefieldname = desc.ShapeFieldName

        # Create search cursor
        #
        rows = arcpy.SearchCursor(infc)

        # Enter for loop for each feature/row
        #

        #Create Cove Point Tuple
        Cove_Point_Info = []
        
        for row in rows:

            #Only write out data for points whose regions exist
            try:
                
                #Assign Parent GUID and flag
                GUID_index = [x[2] for x in Cove_Info].index(row.CoveName)
                GUID = Cove_Info[GUID_index][0]
                
                flag = True
            except:
                flag = False
                

            if flag == True:
                
                # Create the geometry object 'feat'
                #
                feat = row.getValue(shapefieldname)
                pnt = feat.getPart()
               
                #Populate Cove Point Tuple (Cove Name, ID, X, Y, Parent ID)
                Cove_Point_Data = (row.CoveName,
                                   row.ICPR_GUID,
                                   pnt.X,
                                   pnt.Y)
                Cove_Point_Info.append(Cove_Point_Data)
          
    except:
        
        arcpy.GetMessages()
        
    #Empty cursor
    del rows
    
    #Create Cove Table
  
    ####################################################################
    #Create OF Channel Cove CSV
    ####################################################################
        
    try:        
    
        infc = "OF_ChannelControlVolume"
        
        #Clear current selection
        try:
            arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
        except:            
            pass
        
        outfc = outpath + '\\' + stem + "_Feature_OF_ChannelControlVolume_Cove.csv"
       

        # Identify the input feature class to describe
        #
        desc = arcpy.Describe(infc)
        shapefieldname = desc.ShapeFieldName
        
        # Create search cursor
        #
        rows = arcpy.SearchCursor(infc)

        # Write headers to CSV
        a = ["ParentID",
             "ID",
             "Vertex_From",
             "Vertex_To",
             "Link_Connection_X",
             "Link_Connection_Y",
             "Link_Connection_Z"]
        
        with open(outfc,'wb') as f:
            csv_writerow(f,a)        

            # Enter for loop for each feature/row
            for row in rows:
                
                #only write out spatial data for items with overland flow regions in GDB
                flag = False
                if row.OverlandFlowRegion in OFRegionDict:
                    flag = True
                    CCV_PTs = []

                if flag is True:                

                    # Create the geometry object
                    #
                    feat = row.getValue(shapefieldname)
                    partnum = 0

                    # Count the number of points in the current feature
                    #
                    partcount = feat.partCount
                    
                    # Enter while loop for each part in the feature 
                    #
                    while partnum < partcount:
                        # Print the part number
                        #
                        part = feat.getPart(partnum)
                        pnt = part.next()
                        pntcount = 0
                
                        # Enter while loop for each vertex
                        #
                        while pnt:

                            #Add channel CV point data to tuple                            
                            CCV_Data = (pntcount, pnt.X, pnt.Y)
                            CCV_PTs.append(CCV_Data)                                
                            pnt = part.next()
                            pntcount = pntcount + 1

                        # If pnt is null, either the part is finished or there is an 
                            #   interior ring
                            #
                            if not pnt: 
                                pnt = part.next()
                                if pnt:
                                    print("Interior Ring:")
                        partnum = partnum + 1

                        #Cycle through all cove features so none get skipped
                        #Flag Channel CV with Cove
                        try:
                            
                            Cove_index = [x[1] for x in Cove_Info].index(row.Name)
                            Cove_flag = True
                                                    
                        except:
                            arcpy.GetMessages()
                            
                            Cove_flag = False

                    #Cycle through data to write out data
                    if Cove_flag == True:
                        
                        for cove in Cove_Info:

                            if cove[1] == row.Name:

                                ID = cove[0]

                                
                                ParenID = row.ICPR_GUID

                                #Convert points to string for comparisons
                                StartX = cove[3]
                                StartX = str(round(StartX,2))                        
                                
                                StartY = cove[5]
                                StartY = str(round(StartY,2))
                                
                                EndX = cove[4]
                                EndX = str(round(EndX,2))
            
                                
                                EndY = cove[6]
                                EndY = str(round(EndY,2))

                                Drawn = cove[7]

                                #Obtain Cove Link connection data
                                #Get Cove name from Cove link (Cove_Info)
                                Cove_Name = cove[2]

                                #Get x,y info from Code_Point_Info
                                Cove_index = [x[0] for x in Cove_Point_Info].index(Cove_Name)

                                Link_Connection_X = Cove_Point_Info[Cove_index][2]
                                Link_Connection_Y = Cove_Point_Info[Cove_index][3]

                            #Convert CV point to string for comparisons
                            for tup in CCV_PTs:

                                if CCV_PTs[-1][0] == tup[0]:
                                    Vertex = 0
                                else:
                                    Vertex = tup[0]
                                
                                
                                CV_X = str(round(tup[1],2))

                                CV_Y = str(round(tup[2],2))

                                if Cove_flag == True and StartX == CV_X and StartY == CV_Y: 
                                    Vertex_1 = Vertex
                                
                                if Cove_flag == True and EndX == CV_X and EndY == CV_Y:
                                    Vertex_2 = Vertex 

                            #Determine direction CV is drawn                        
                            n = pntcount            
                            i = 0
                            SUM = 0

                            while i < n:
                                
                                if i < n-1:
                                    
                                    Xi0 = CCV_PTs[i][1]
                                    Xi1 = CCV_PTs[i+1][1]
                                    Yi0 = CCV_PTs[i][2]
                                    Yi1 = CCV_PTs[i+1][2]

                                    delta_SUM = (Xi1-Xi0)*(Yi1+Yi0)
                                    SUM = SUM + delta_SUM

                                    i = i + 1

                                if i == n-1:
                                
                                    Xi0 = CCV_PTs[i][1]
                                    Xi1 = CCV_PTs[0][1]
                                    Yi0 = CCV_PTs[i][2]
                                    Yi1 = CCV_PTs[0][2]

                                    delta_SUM = (Xi1-Xi0)*(Yi1+Yi0)
                                    SUM = SUM + delta_SUM
                                    i = i + 1

                            #Determine direction
                            if SUM < 0:
                                Direction = "COUNTERCLOCKWISE"
                            else:
                                Direction = "CLOCKWISE"

                            #Assign Vertex assignment based on direction the direction
                            #each feature was drawn
                            if Direction != Drawn:

                                Vertex_From = Vertex_2
                                Vertex_To = Vertex_1
                                
                            else:

                                Vertex_From = Vertex_1
                                Vertex_To = Vertex_2

                            #Only write out data once when names match
                            if cove[1] == row.Name:                       

                                a = [ParenID,
                                     ID,
                                     Vertex_From,
                                     Vertex_To,
                                     Link_Connection_X,
                                     Link_Connection_Y,
                                     ""]                    

                                csv_writerow(f,a)

        del rows

    except:
        arcpy.GetMessages()


######################################################################################################
######################################################################################################
