#Copyright 2020 Streamline Technologies, Inc.
#All Rights Reserved
#Name:  Frank W. McKinnie, PE, CFM
#Company: Streamline Technologies, Inc.
#Purpose:  Writes CSV for import into ICPR4 from GWIS version 2.1 GDB

def Node_write_file(outpath, ScDict, stem, NodeTypeConvert, NodeProcessed):
    #Add forward slash for file path

    slash = "\\"
    import arcpy, os, uuid
    from iccsv import csv_writerow

    #Add global value for node type
    NodeDict = {}

    ####################################################################
    #Write node feature data to CSV
    ####################################################################

    try:
        infc = "ICPR_NODE"
        outfc = outpath + slash + stem + "_Node.csv"

        #Clear current selection
        try:
            arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
        except:
            pass

        # Identify the input feature class to describe
        #
        desc = arcpy.Describe(infc)
        shapefieldname = desc.ShapeFieldName

        # Create search cursor
        #
        rows = arcpy.SearchCursor(infc)

        # Write headers to CSV
        a = ["ID",
             "ParentID",
             "BaseFlow",
             "BoundaryStage",
             "Comment",
             "InitialStage",
             "IsPlaced",
             "Name",
             "ShapeX",
             "ShapeY",
             "ShapeZ",
             "TextX",
             "TextY",
             "TextZ",
             "Type",
             "WarningStage",
             "AlertStage",
             "WQ_WQOption",
             "WQ_Comment",
             "WQ_ShapeX",
             "WQ_ShapeY",
             "WQ_ShapeZ",
             "WQ_IsPlaced"]

        with open(outfc,'wb') as f:
            csv_writerow(f,a)

            # Enter for loop for each feature/row
            #
            for row in rows:
                # Create the geometry object 'feat'
                #
                feat = row.getValue(shapefieldname)

                try:
                    pnt = feat.getPart()
                except:
                    pass

                #Convert domain values for node type field & also create tuple with node name, guid and type
                ParaList = []

                #specify ICPR4 node type code
                try:

                    #Get the index value for node type conversion
                    NodeTypeConvert_index = [x[1] for x in NodeTypeConvert].index(int(row.TYPE))

                    #Assign ICPR4 code value
                    ICPR4TypeCode = NodeTypeConvert[NodeTypeConvert_index][2]

                #If type code is not supported, end routine
                except:

                    arcpy.AddMessage("Node Type Not Supported: " + row.NAME)
                    arcpy.AddMessage("Process Aborted")
                    os.remove(outfc)
                    exit


                #Assign ID and ParentID
                GUID = ScDict[row.SCENARIO][0]

                #Define Comment
                Comment = row.COMMENT
                CommentWQ = row.WQ_Comment

                #Check if node has geometry
                try:
                    PntX = pnt.X
                    PntY = pnt.Y
                    IsPlaced = row.ISPLACED

                #If geometry doesn't exist
                except:

                    PntX = 0
                    PntY = 0
                    IsPlaced = 0

                #Write x, y coordinates of current point to CSV
                a = [row.ICPR_GUID,
                     GUID,
                     row.BASE_FLOW,
                     row.BOUNDARY_STAGE,
                     Comment,
                     row.INITIAL_STAGE,
                     IsPlaced,
                     row.NAME,
                     PntX,
                     PntY,
                     0,
                     PntX,
                     PntY,
                     0,
                     ICPR4TypeCode,
                     row.WARNING_STAGE,
                     row.AlertStage,
                     row.WQ_WQOption,
                     CommentWQ,
                     row.WQ_ShapeX,
                     row.WQ_ShapeY,
                     row.WQ_ShapeZ,
                     row.WQ_IsPlaced]

                csv_writerow(f,a)

                #store node type and other data as tuple
                Node_Values = (row.SCENARIO,
                               row.NAME,
                               ICPR4TypeCode,
                               row.ICPR_GUID,
                               PntX,
                               PntY)

                NodeDict[row.NAME]=Node_Values

        del rows

    except:

        arcpy.GetMessages()

    ####################################################################
    #Write node WQ feature data to CSV
    ####################################################################

    if NodeProcessed == True:

        try:
            infc = "ICPR4_Node_WQ"
            outfc = outpath + slash + stem + "_Node_WQ.csv"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            # Create search cursor
            #
            rows = arcpy.SearchCursor(infc)

            # Write headers to CSV
            a = ["ID",
                 "ParentID",
                 "Constituent",
                 "Concentration",
                 "InitialConcentration",
                 "RemovalEfficiency",
                 "UseDfltInitConcentration",
                 "UseDfltIrrConcentration",
                 "IrreducibleConcentration"]

            with open(outfc,'wb') as f:
                csv_writerow(f,a)

                # Enter for loop for each feature/row
                #

                for row in rows:

                    #Write Node GUID as Parent ID
                    Node_GUID=NodeDict[row.Node_Name][3]


                    #Write data to CSV
                    a = [row.ICPR_GUID,
                         Node_GUID,
                         row.Constituent,
                         row.Concentration,
                         row.InitialConcentration,
                         row.RemovalEfficiency,
                         row.UseDfltInitConcentration,
                         row.UseDfltIrrConcentration,
                         row.IrreducibleConcentration]

                    csv_writerow(f,a)

            del rows
        except:
            arcpy.GetMessages()

    ####################################################################
    #Write node external hydrograph feature data to CSV
    ####################################################################

    if NodeProcessed == True:

        try:
            infc = "ICPR4_Node_External_Hydrograph"
            outfc = outpath + slash + stem + "_Node_ExternalHydrograph.csv"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            # Create search cursor
            #
            rows = arcpy.SearchCursor(infc)

            # Write headers to CSV
            a = ["ID","ParentID", "ExternalHydrograph"]
            with open(outfc,'wb') as f:
                csv_writerow(f,a)

                # Enter for loop for each feature/row
                #

                for row in rows:

                    #Write Node GUID as Parent ID
                    Node_GUID=NodeDict[row.NodeName][3]


                    #Write data to CSV
                    a = [row.ICPR_GUID, Node_GUID, row.Name]

                    csv_writerow(f,a)

            del rows
        except:
            arcpy.GetMessages()

    ####################################################################
    #Write node external hydrograph WQ feature data to CSV
    ####################################################################

    if NodeProcessed == True:

        try:
            infc = "ICPR4_Node_ExternalHydrograph_WQ"
            outfc = outpath + slash + stem + "_Node_ExternalHydrograph_WQ.csv"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            # Create search cursor
            #
            rows = arcpy.SearchCursor(infc)

            # Write headers to CSV
            a = ["ID",
                 "ParentID",
                 "ExternalHydrograph",
                 "Constituent",
                 "Concentration"]

            with open(outfc,'wb') as f:
                csv_writerow(f,a)

                # Enter for loop for each feature/row
                #

                for row in rows:

                    #Write Node GUID as Parent ID
                    Node_GUID=NodeDict[row.Node_Name][3]


                    #Write data to CSV
                    a = [row.ICPR_GUID,
                         Node_GUID,
                         row.ExternalHydrograph,
                         row.Constituent,
                         row.Concentration]

                    csv_writerow(f,a)

            del rows
        except:
            arcpy.GetMessages()



        ####################################################################
        #Write node storage table data to CSV
        ####################################################################

        #Process node storage data
        try:
            infc = "ICPR_NODE_STORAGE"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            #Define output location.
            outfc = outpath + slash + stem + "_Node_Point.csv"

            # Create new search cursor
            #
            rows = arcpy.SearchCursor(infc,sort_fields="NAME; STAGE_VAL")

            # Write headers to CSV
            a = ["ID","ParentID", "Day", "Month", "Year", "X","Y"]
            with open(outfc,'wb') as f:
                csv_writerow(f,a)

                # Enter for loop for each feature/row
                for row in rows:

                    try:

                        #Write Node GUID as Parent ID
                        #Also determine node type
                        Node_GUID = NodeDict[row.NAME][3]
                        Node_Type = NodeDict[row.NAME][2]

                        #Stage area nodes
                        if Node_Type == 2:

                            a = [row.ICPR_GUID, Node_GUID, "", "", "", row.STAGE_VAL,row.AREA_MS]
                            csv_writerow(f,a)

                        #Stage volume nodes
                        if Node_Type == 6:

                            a = [row.ICPR_GUID, Node_GUID, "", "", "", row.STAGE_VAL,row.VOLUME_VAL]
                            csv_writerow(f,a)
                    except:
                        arcpy.GetMessages()

            del rows

        except:
            arcpy.GetMessages()


        ####################################################################
        #Write node time stage data to CSV
        ####################################################################

        #Process node storage data
        try:
            infc = "ICPR_NODE_TIMESTAGE"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            #Define output location.
            outfc = outpath + slash + stem + "_Node_Point.csv"

            # Create new search cursor
            #
            rows = arcpy.SearchCursor(infc,sort_fields="NAME; YEAR; MONTH; DAY; TIME_VAL")

            #append because we already added header and Z/A pts to this file
            with open(outfc,'ab') as f:

                # Enter for loop for each feature/row
                for row in rows:

                    try:

                        #Write Node GUID as Parent ID
                        #Also determine node type
                        Node_GUID = NodeDict[row.NAME][3]
                        Node_Type = NodeDict[row.NAME][2]


                        #Stage area nodes
                        if Node_Type == 4:

                            a = [row.ICPR_GUID, Node_GUID, row.DAY, row.MONTH, row.YEAR, row.TIME_VAL,row.STAGE_VAL]
                            csv_writerow(f,a)

                    except:
                        arcpy.GetMessages()

            del rows

        except:
            arcpy.GetMessages()

        ####################################################################
        #Write node storage polygon data to CSV
        ####################################################################

        #Process node storage data
        try:
            infc = "NODE_STORAGE_POLYGON"

            #Clear current selection
            try:
                arcpy.SelectLayerByAttribute_management(infc, "CLEAR_SELECTION")
            except:
                pass

            #Define output location.
            outfc = outpath + slash + stem + "_Node_Polygon_Point.csv"

            # Create new search cursor
            #
            rows = arcpy.SearchCursor(infc)

            # Write headers to CSV
            a = ["ID","ParentID", "Vertex", "X", "Y"]
            with open(outfc,'wb') as f:
                csv_writerow(f,a)

                # Enter for loop for each feature/row
                for row in rows:

                    try:

                        # Create the geometry object
                        #
                        feat = row.getValue(shapefieldname)
                        partnum = 0

                        # Count the number of points in the current multipart feature
                        #
                        partcount = feat.partCount

                        #Total number of points in feature
                        pointcount = feat.pointCount

                        # Enter while loop for each part in the feature (if a singlepart feature
                        # this will occur only once)
                        #
                        while partnum < partcount:
                            # Print the part number
                            #
                            part = feat.getPart(partnum)
                            pnt = part.next()
                            pntcount = 0

                            #Write Node GUID as Parent ID
                            #Also determine node type
                            Node_GUID = NodeDict[row.NODENAME][3]

                            # Enter while loop for each vertex
                            #
                            while pnt:
                                #Write x, y coordinates of current point to CSV
                                a = ["{"+str(uuid.uuid4())+"}",Node_GUID,  pntcount, pnt.X, pnt.Y, ""]

                                #Only write points up the last point, which is excluded
                                if pntcount < pointcount-1:
                                    csv_writerow(f,a)

                                pnt = part.next()
                                pntcount = pntcount + 1

                                # If pnt is null, either the part is finished or there is an
                                #   interior ring
                                #
                                if not pnt:
                                    pnt = part.next()
                                    if pnt:
                                        print("Interior Ring:")
                            partnum = partnum + 1

                    except:
                        pass

            del rows

        except:
            arcpy.GetMessages()

    #############################################################################################
    #Write out returned values (row.SCENARIO, row.NAME, ICPR4TypeCode,row.ICPR_GUID, pnt.X, pnt.Y)
    #############################################################################################
    try:

        return NodeDict

    except:
        arcpy.GetMessages()
######################################################################################################
######################################################################################################
